/**
* \file: EndpointDataSharing.cpp
*
* \version: 0.4
*
* \release: $Name:$
*
* Transmission mechanism to share data between endpoints
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <cerrno>
#include <uspi_macros.h>
#include <uspi/EndpointDataSharing.h>

namespace adit { namespace uspi {
using std::make_pair;
using std::map;

SharedDataSender::SharedDataSender()
{
    mReceiversListMutex = PTHREAD_MUTEX_INITIALIZER;
}

SharedDataSender::~SharedDataSender() { }

int SharedDataSender::subscribeForDataSharing(SharedDataReceiver& inReceiver)
{
    uspi_return_value_on_invalid_argument(uspi, inReceiver.sessionCtx() == nullptr, EINVAL);

    int ret_val = pthread_mutex_lock(&mReceiversListMutex);
    if (ret_val == 0)
    {
        auto tmp_pair = mReceiversList.insert(make_pair(inReceiver.sessionCtx(), &inReceiver));

        ret_val = pthread_mutex_unlock(&mReceiversListMutex);

        if (ret_val == 0 && tmp_pair.second == false)
        {
            /* SessionContext was already subscribed, return -1 */
            ret_val = -1;
        }
    }
    return ret_val;
}

int SharedDataSender::unsubscribeFromDataSharing(SharedDataReceiver& inReceiver)
{
    uspi_return_value_on_invalid_argument(uspi, inReceiver.sessionCtx() == nullptr, EINVAL);

    int ret_val = pthread_mutex_lock(&mReceiversListMutex);
    if (ret_val == 0)
    {
        auto num_of_erased = mReceiversList.erase(inReceiver.sessionCtx());

        ret_val = pthread_mutex_unlock(&mReceiversListMutex);

        if (ret_val == 0 && num_of_erased == 0)
        {
            /* SessionContext was already unsubscribed, return -1 */
            ret_val = -1;
        }
    }
    return ret_val;
}

int SharedDataSender::transmitSurfaceData(void* inSessionCtx, wl_display* inDisplay)
{
    uspi_return_value_on_invalid_argument(uspi, inSessionCtx == nullptr, EINVAL);
    uspi_return_value_on_invalid_argument(uspi, inDisplay == nullptr, EINVAL);

    bool rcvr_found = true;

    int ret_val = pthread_mutex_lock(&mReceiversListMutex);
    if (ret_val == 0)
    {
        SharedDataReceiver* receiver = findReceiverBySessionCtx(inSessionCtx);
        if (receiver == nullptr)
        {
            /* Given SessionContext is not registered for transmission */
            rcvr_found = false;
        } else {
            receiver->onSurfaceDataProvided(inDisplay);
        }

        ret_val = pthread_mutex_unlock(&mReceiversListMutex);

        if (ret_val == 0 && !rcvr_found)
        {
            /* If everything with the mutexes worked error free but there is no receiver registered, return -1 */
            ret_val = -1;
        }
    }
    return ret_val;
}

int SharedDataSender::notifySurfaceDataExpiration(void* inSessionCtx)
{
    uspi_return_value_on_invalid_argument(uspi, inSessionCtx == nullptr, EINVAL);

    bool rcvr_found = true;

    int ret_val = pthread_mutex_lock(&mReceiversListMutex);
    if (ret_val == 0)
    {
        SharedDataReceiver* receiver = findReceiverBySessionCtx(inSessionCtx);
        if (receiver == nullptr)
        {
            /* Given SessionContext is not registered for transmission */
            rcvr_found = false;
        } else {
            receiver->onSurfaceDataExpired();
        }

        ret_val = pthread_mutex_unlock(&mReceiversListMutex);

        if (ret_val == 0 && !rcvr_found)
        {
            /* If everything with the mutexes worked error free but there is no receiver registered, return -1 */
            ret_val = -1;
        }
    }
    return ret_val;
}

int SharedDataSender::transmitResolutionData(void* inSessionCtx, int inWidth, int inHeight, int inMarginX, int inMarginY)
{
    uspi_return_value_on_invalid_argument(uspi, inSessionCtx == nullptr, EINVAL);

    bool rcvr_found = true;

    int ret_val = pthread_mutex_lock(&mReceiversListMutex);
    if (ret_val == 0)
    {
        SharedDataReceiver* receiver = findReceiverBySessionCtx(inSessionCtx);
        if (receiver == nullptr)
        {
            /* Given SessionContext is not registered for transmission */
            rcvr_found = false;
        } else {
            receiver->onResolutionDataProvided(inWidth, inHeight, inMarginX, inMarginY);
        }

        ret_val = pthread_mutex_unlock(&mReceiversListMutex);

        if (ret_val == 0 && !rcvr_found)
        {
            /* If everything with the mutexes worked error free but there is no receiver registered, return -1 */
            ret_val = -1;
        }
    }
    return ret_val;
}

SharedDataReceiver* SharedDataSender::findReceiverBySessionCtx(void* inSessionCtx)
{
    uspi_return_value_on_invalid_argument(uspi, inSessionCtx == nullptr, nullptr);

    map<void*, SharedDataReceiver*>::iterator it = mReceiversList.find(inSessionCtx);
    if (it != mReceiversList.end())
    {
        return it->second;
    }

    /* Cannot find the SessionContext in the list */
    return nullptr;
}

} } /* namespace adit { namespace uspi { */
